# This code is free to use.

import bpy

try:
    from mido import MidiFile
except ImportError:
    print("This module depends on Mido for MIDI support. Please make sure you have this module installed.")

print("AniMIDI has been successfully imported.")

def get_current_value(target_object, data_path, index=None):
    if index == None:
        return eval(f"target_object.{data_path}")
    else:
        return eval(f"target_object.{data_path}[{index}]")

def insert_keyframe(target_object=None, data_path=None, index=None, frame=None, value=None, handle_left=None, handle_left_type=None, handle_right=None, handle_right_type=None, argh=None):

    if target_object is None: raise ValueError
    if data_path is None: raise ValueError
    if frame is None: raise ValueError
    if value is None: raise ValueError

    # find fcurve
    if target_object.animation_data is None:
        target_object.animation_data_create()
    a_d = target_object.animation_data

    if a_d.action is None:
        a_d.action = bpy.data.actions.new("anim_data")

    if index == None:
        fc = a_d.action.fcurves.find(data_path)
        if fc is None:
            fc = a_d.action.fcurves.new(data_path)
    else:
        fc = a_d.action.fcurves.find(data_path, index=index)
        if fc is None:
            fc = a_d.action.fcurves.new(data_path, index=index)

    if value == "current":
        new_frame = fc.keyframe_points.insert(frame, value=get_current_value(target_object, data_path, index))
    else:
        new_frame = fc.keyframe_points.insert(frame, value)

    if handle_left is not None:
        new_frame.handle_left = handle_left
    if handle_left_type is not None:
        new_frame.handle_left_type = handle_left_type
    if handle_right is not None:
        new_frame.handle_right = handle_right
    if handle_right_type is not None:
        new_frame.handle_right_type = handle_right_type

    return new_frame


def change_notes_to_absolute_time(track):
    mem1 = 0
    for message in track:
        message.time = message.time + mem1
        mem1 = message.time
    return track


def convert_time_to_frame(time, animation_fps, animation_time_scale, animation_time_offset):
    if time is None: raise ValueError
    return ((time / 1000) * animation_fps * animation_time_scale) + animation_time_offset


def get_message(base_message = None, track_notes = None, direction = "next", type = "note_on", note = None, time_before_all_other_notes = -100, time_after_all_other_notes = 100000):
    if base_message is None: raise ValueError
    #if track_notes is None: raise ValueError
    if not (direction == "next" or direction == "previous"): raise ValueError

    message_index = track_notes.index(base_message)

    if direction == "previous":
        for i in range(message_index, 0, -1):
            #print(i-1)
            compare_message = track_notes[i-1]
            if compare_message.type == type:
            #if compare_message.time < base_message.time:
                if note == None:
                    return (compare_message, abs(compare_message.time - base_message.time))
                else:
                    if compare_message.note == note:
                        return (compare_message, abs(compare_message.time - base_message.time))
        return (None, abs(time_before_all_other_notes - base_message.time))

    elif direction == "next":
        #print(len(track_notes))
        for i in range(message_index, len(track_notes) - 1):
            #print(i+1)
            compare_message = track_notes[i+1]
            if compare_message.type == type:
            #if compare_message.time > base_message.time:
                #print(compare_message.type)
                #print(type)
                if note == None:
                    return (compare_message, abs(compare_message.time - base_message.time))
                else:
                    if compare_message.note == note:
                        return (compare_message, abs(compare_message.time - base_message.time))
        return (None, abs(time_after_all_other_notes - base_message.time))

# This returns a list of all the messages in the track, but with the times adjusted to frame numbers
def load_track(midifile, track_index, animation_fps, animation_time_scale, animation_time_offset):

    mid = MidiFile(midifile, clip=True)
    track = mid.tracks[track_index]

    track = change_notes_to_absolute_time(track)

    #change all the message times to frame numbers
    for message in track:
        message.time = convert_time_to_frame(message.time, animation_fps, animation_time_scale, animation_time_offset)

    return track

def scale_linear_maxmin(val, x1, x2, y1, y2):
    return (((y2 - y1) / (x2 - x1)) * (val - x1)) + y1

# A function designed for vibrating strings
def generate_sine(starting_frame=0, noteoff_frame=0, ending_frame=0, period=0, delay=0, attack=0, hold=0, decay=0, sustain=0, release=0, vibration_scale=1, scene=None):
    # Argument sanity
    if starting_frame >= ending_frame: raise ValueError
    if noteoff_frame < starting_frame: raise ValueError
    if period <= 0: raise ValueError

    # Change ADSR values to global time
    delay += starting_frame
    attack += delay
    hold += attack
    decay += hold
    release += decay
    #print(release)

    #Debugging function
    def insert_marker(value, frame):
        if scene is not None:
            scene.timeline_markers.new(value, frame=frame)

    insert_marker("Start", starting_frame)
    insert_marker("Noteoff", noteoff_frame)
    insert_marker("Delay", delay)
    insert_marker("Attack", attack)
    insert_marker("Hold", hold)
    insert_marker("Decay", decay)
    insert_marker("Release", release)

    # Make the sine list with the ADSR values
    sine_list = {}
    count = 0
    while 1:
        frame_num = (count * period) + starting_frame
        #print(frame_num)
        sine_val = 0
        if frame_num >= ending_frame:
            break
        # Post_release
        if frame_num >= release:
            #print("AHOOEY!")
            sine_val = 0
        # During release
        elif frame_num >= noteoff_frame:
            if noteoff_frame < decay:
                value_at_noteoff = scale_linear_maxmin(noteoff_frame, hold, decay, 1, sustain)
                sine_val = scale_linear_maxmin(frame_num, noteoff_frame, noteoff_frame + (release - decay), value_at_noteoff, 0)
                if sine_val < 0:
                    sine_val = 0
            else:
                sine_val = scale_linear_maxmin(frame_num, noteoff_frame, release, sustain, 0)
        # During sustain
        elif frame_num >= decay:
            sine_val = sustain
        # During decay
        elif frame_num >= hold:
            sine_val = scale_linear_maxmin(frame_num, hold, decay, 1, sustain)
        # During hold
        elif frame_num >= attack:
            sine_val = 1
        # During attack
        elif frame_num >= delay:
            sine_val = scale_linear_maxmin(frame_num, delay, attack, 0, 1)
        # During pre-delay
        else:
            sine_val = 0

        if count % 2 == 1:
            sine_val *= -1
        sine_list[frame_num] = (sine_val * vibration_scale)
        count += 1

    return sine_list

def pluck_string(target_object, data_path, starting_frame=0, noteoff_frame=0, ending_frame=0, period=0, delay=0, attack=0, hold=0, decay=0, sustain=0, release=0, vibration_scale=1, scene=None):
    string_motion = generate_sine(
        starting_frame = starting_frame,
        noteoff_frame = noteoff_frame,
        ending_frame = ending_frame,
        period = period,
        delay = delay,
        attack = attack,
        hold = hold,
        decay = decay,
        sustain = sustain,
        release = release,
        vibration_scale = vibration_scale,
        scene = scene
    )

    insert_keyframe(
        target_object = target_object,
        data_path = data_path,
        frame = starting_frame - 1,
        value = 0
    )

    for frame in string_motion:
        insert_keyframe(
            target_object = target_object,
            data_path = data_path,
            frame = int(frame),
            value = string_motion[frame]
        )

    insert_keyframe(
        target_object = target_object,
        data_path = data_path,
        frame = int(ending_frame),
        value = 0
    )


def order_track(track):
    def get_key(message):
        return message.time
    return sorted(track, key=get_key)

def merge_tracks(track1, track2):
    return order_track(track1 + track2)

def split_track_by_note(track):
    return_dict = {}
    return_dict["not notes"] = []
    for message in track:
        if not hasattr(message, "note"):
            return_dict["not notes"].append(message)
            continue
        if message.note not in return_dict.keys():
            return_dict[message.note] = []
        return_dict[message.note].append(message)
    return return_dict

def filter_only_noteons(track):
    return_list = []
    for message in track:
        if message.type == "note_on":
            return_list.append(message)
    return return_list
